//
//  HealthKitServiceTests.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import XCTest
import HealthKit
@testable import death_app_Watch_App

final class HealthKitServiceTests: XCTestCase {
    
    var healthKitService: HealthKitService!
    var mockHealthStore: MockHKHealthStore!
    
    override func setUpWithError() throws {
        super.setUp()
        mockHealthStore = MockHKHealthStore()
        healthKitService = HealthKitService(healthStore: mockHealthStore)
    }
    
    override func tearDownWithError() throws {
        healthKitService = nil
        mockHealthStore = nil
        super.tearDown()
    }
    
    // MARK: - Authorization Tests
    
    func testHealthKitAvailability() throws {
        let isAvailable = HKHealthStore.isHealthDataAvailable()
        XCTAssertTrue(isAvailable, "HealthKit should be available on watchOS")
    }
    
    func testRequestAuthorizationSuccess() throws {
        let expectation = XCTestExpectation(description: "Authorization success")
        mockHealthStore.shouldSucceedAuthorization = true
        
        healthKitService.requestAuthorization { success, error in
            XCTAssertTrue(success, "Authorization should succeed")
            XCTAssertNil(error, "No error should be returned")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testRequestAuthorizationFailure() throws {
        let expectation = XCTestExpectation(description: "Authorization failure")
        mockHealthStore.shouldSucceedAuthorization = false
        mockHealthStore.authorizationError = NSError(domain: "TestError", code: 1, userInfo: nil)
        
        healthKitService.requestAuthorization { success, error in
            XCTAssertFalse(success, "Authorization should fail")
            XCTAssertNotNil(error, "Error should be returned")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testAuthorizationStatus() throws {
        mockHealthStore.mockAuthorizationStatus = .sharingAuthorized
        
        let status = healthKitService.authorizationStatus(for: HKQuantityType.quantityType(forIdentifier: .heartRate)!)
        XCTAssertEqual(status, .sharingAuthorized, "Should return correct authorization status")
    }
    
    // MARK: - Data Fetching Tests
    
    func testFetchHeartRateData() throws {
        let expectation = XCTestExpectation(description: "Fetch heart rate data")
        
        // Setup mock data
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        let heartRateUnit = HKUnit(from: "count/min")
        let heartRateQuantity = HKQuantity(unit: heartRateUnit, doubleValue: 75.0)
        let heartRateSample = HKQuantitySample(
            type: heartRateType,
            quantity: heartRateQuantity,
            start: Date().addingTimeInterval(-3600),
            end: Date()
        )
        
        mockHealthStore.mockSamples = [heartRateSample]
        
        healthKitService.fetchHeartRateData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(error, "No error should occur")
            XCTAssertNotNil(samples, "Samples should be returned")
            XCTAssertEqual(samples?.count, 1, "Should return one sample")
            
            if let sample = samples?.first as? HKQuantitySample {
                XCTAssertEqual(sample.quantity.doubleValue(for: heartRateUnit), 75.0, "Heart rate should match")
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testFetchStepCountData() throws {
        let expectation = XCTestExpectation(description: "Fetch step count data")
        
        let stepType = HKQuantityType.quantityType(forIdentifier: .stepCount)!
        let stepQuantity = HKQuantity(unit: HKUnit.count(), doubleValue: 8500.0)
        let stepSample = HKQuantitySample(
            type: stepType,
            quantity: stepQuantity,
            start: Calendar.current.startOfDay(for: Date()),
            end: Date()
        )
        
        mockHealthStore.mockSamples = [stepSample]
        
        healthKitService.fetchStepCountData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(error)
            XCTAssertNotNil(samples)
            XCTAssertEqual(samples?.count, 1)
            
            if let sample = samples?.first as? HKQuantitySample {
                XCTAssertEqual(sample.quantity.doubleValue(for: .count()), 8500.0)
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testFetchActiveEnergyData() throws {
        let expectation = XCTestExpectation(description: "Fetch active energy data")
        
        let energyType = HKQuantityType.quantityType(forIdentifier: .activeEnergyBurned)!
        let energyQuantity = HKQuantity(unit: .kilocalorie(), doubleValue: 450.0)
        let energySample = HKQuantitySample(
            type: energyType,
            quantity: energyQuantity,
            start: Date().addingTimeInterval(-3600),
            end: Date()
        )
        
        mockHealthStore.mockSamples = [energySample]
        
        healthKitService.fetchActiveEnergyData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(error)
            XCTAssertNotNil(samples)
            
            if let sample = samples?.first as? HKQuantitySample {
                XCTAssertEqual(sample.quantity.doubleValue(for: .kilocalorie()), 450.0)
            }
            
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    // MARK: - Background Delivery Tests
    
    func testEnableBackgroundDelivery() throws {
        let expectation = XCTestExpectation(description: "Enable background delivery")
        mockHealthStore.shouldSucceedBackgroundDelivery = true
        
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        
        healthKitService.enableBackgroundDelivery(for: heartRateType, frequency: .immediate) { success, error in
            XCTAssertTrue(success, "Background delivery should be enabled")
            XCTAssertNil(error)
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testDisableBackgroundDelivery() throws {
        let expectation = XCTestExpectation(description: "Disable background delivery")
        
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        
        healthKitService.disableBackgroundDelivery(for: heartRateType) { success, error in
            XCTAssertTrue(success, "Background delivery should be disabled")
            XCTAssertNil(error)
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    // MARK: - Observer Query Tests
    
    func testStartObserving() throws {
        let expectation = XCTestExpectation(description: "Start observing")
        
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        
        healthKitService.startObserving(sampleType: heartRateType) { query, completionHandler, error in
            XCTAssertNotNil(query, "Query should be created")
            XCTAssertNil(error)
            expectation.fulfill()
            completionHandler()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testStopObserving() throws {
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        
        // First start observing
        healthKitService.startObserving(sampleType: heartRateType) { _, completionHandler, _ in
            completionHandler()
        }
        
        // Then stop observing
        healthKitService.stopObserving(sampleType: heartRateType)
        
        // Verify that query was stopped (mock should track this)
        XCTAssertTrue(mockHealthStore.stoppedQueries.count > 0, "Should have stopped queries")
    }
    
    // MARK: - Error Handling Tests
    
    func testNetworkErrorHandling() throws {
        let expectation = XCTestExpectation(description: "Network error handling")
        
        mockHealthStore.shouldFailWithNetworkError = true
        
        healthKitService.fetchHeartRateData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(samples)
            XCTAssertNotNil(error)
            XCTAssertEqual((error as NSError?)?.code, 5, "Should handle network error")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testPermissionDeniedHandling() throws {
        let expectation = XCTestExpectation(description: "Permission denied handling")
        
        mockHealthStore.shouldFailWithPermissionError = true
        
        healthKitService.fetchStepCountData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(samples)
            XCTAssertNotNil(error)
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    // MARK: - Data Validation Tests
    
    func testInvalidDateRange() throws {
        let expectation = XCTestExpectation(description: "Invalid date range")
        
        let futureDate = Date().addingTimeInterval(86400) // Tomorrow
        let pastDate = Date().addingTimeInterval(-86400) // Yesterday
        
        healthKitService.fetchHeartRateData(from: futureDate, to: pastDate) { samples, error in
            XCTAssertNil(samples)
            XCTAssertNotNil(error)
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testEmptyDataHandling() throws {
        let expectation = XCTestExpectation(description: "Empty data handling")
        
        mockHealthStore.mockSamples = [] // No samples
        
        healthKitService.fetchHeartRateData(from: Date().addingTimeInterval(-86400), to: Date()) { samples, error in
            XCTAssertNil(error)
            XCTAssertNotNil(samples)
            XCTAssertEqual(samples?.count, 0, "Should handle empty data gracefully")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    // MARK: - Performance Tests
    
    func testLargeDataSetPerformance() throws {
        // Create large mock dataset
        var largeSamples: [HKQuantitySample] = []
        let heartRateType = HKQuantityType.quantityType(forIdentifier: .heartRate)!
        let heartRateUnit = HKUnit(from: "count/min")
        
        for i in 0..<10000 {
            let quantity = HKQuantity(unit: heartRateUnit, doubleValue: Double(60 + i % 40))
            let sample = HKQuantitySample(
                type: heartRateType,
                quantity: quantity,
                start: Date().addingTimeInterval(-Double(i * 60)),
                end: Date().addingTimeInterval(-Double(i * 60) + 30)
            )
            largeSamples.append(sample)
        }
        
        mockHealthStore.mockSamples = largeSamples
        
        let expectation = XCTestExpectation(description: "Large dataset performance")
        
        let startTime = CFAbsoluteTimeGetCurrent()
        
        healthKitService.fetchHeartRateData(from: Date().addingTimeInterval(-604800), to: Date()) { samples, error in
            let endTime = CFAbsoluteTimeGetCurrent()
            let timeElapsed = endTime - startTime
            
            XCTAssertLessThan(timeElapsed, 1.0, "Should process large dataset within 1 second")
            XCTAssertEqual(samples?.count, 10000)
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
    }
}